//
//  $Id: AHKEmail.m 98 2009-06-12 17:01:44Z fujidana $
//  Copyright (c) 2005-2009 Fujidana All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#import "AHKMail.h"
#import "GSNSDataExtensions.h"
#import <string.h>


@interface AHKEmail (Private)

- (BOOL)parseData;

@end


@interface NSString (FJDNMIMEEncode)

- (NSString *)stringByDecodingRFC2047MessageHeader;

@end


@implementation AHKEmail

// --- Designated initializer
- (id)initWithData:(NSData *)datum box:(int)aBox number:(int)aNumber status:(int)aStatus isSecured:(BOOL)flag
{
	self = [super initWithData:datum
						   box:aBox
						number:aNumber
						status:aStatus
					 isSecured:flag];
	if (self != nil)
	{
		headerDictionary = nil;
		body = nil;
	}
	return self;
}

- (void)dealloc
{
	[headerDictionary release];
	[body release];
	
	[super dealloc];
}

- (NSString *)from
{
	return [[self headerDictionary] objectForKey:@"From"];
}

- (NSString *)to
{
	return [[self headerDictionary] objectForKey:@"To"];
}

- (NSDate *)date
{
	NSString *dateString = [[self headerDictionary] objectForKey:@"Date"];
	if (dateString)
	{
		return [NSDate dateWithNaturalLanguageString:dateString];
	}
	return nil;
}

- (NSData *)message
{
	return [self data];
}

- (NSString *)subject
{
	return [[[self headerDictionary] objectForKey:@"Subject"] stringByDecodingRFC2047MessageHeader];
}

- (NSString *)body
{
	//	return [[[NSString alloc] initWithData:[self data] encoding:NSISO2022JPStringEncoding] autorelease];
	if (body == nil)
	{
		[self parseData];
	}
	return body;
}

// set date, from, subject
- (NSDictionary *)headerDictionary
{
	if (headerDictionary == nil)
	{
		[self parseData];
	}
	return headerDictionary;
}

- (NSString *)object
{
	if ([self box] == AHKSentMailBox || [self box] == AHKDraftMailBox)
	{
		return [self to];
	}
	else
	{
		return [self from];
	}
}

#pragma mark private methods

- (BOOL)parseData
{
	// Separate header and body at "\r\n\r\n"
	
	int dataLength = [[self data] length];
	
	char buffer[dataLength + 1];
	[[self data] getBytes:buffer];
	buffer[dataLength] = '\0';
	
	char *splitPoint = strstr(buffer,"\r\n\r\n");
	if (splitPoint == NULL)
	{
		return NO;
	}
	
	int headerLength = splitPoint - buffer;
	
	NSData *headerData = [[self data] subdataWithRange:NSMakeRange(0, headerLength)];
	NSData *bodyData = [[self data] subdataWithRange:NSMakeRange(headerLength + 4, dataLength - (headerLength + 4))];
	NSString *headerString = [[NSString alloc] initWithData:headerData encoding:NSShiftJISStringEncoding];	// Text encoded by ISO-2022-JP, which is general text encoding, is safely interpretted by Shift-JIS.
	
	// In case failed to interpret text (probablly messageString contains EMOJI)
	if (headerString == nil)
	{
		return NO;
	}
	
	NSScanner *mailScanner = [NSScanner scannerWithString:headerString];
	[mailScanner setCharactersToBeSkipped:[NSCharacterSet whitespaceCharacterSet]];
	NSString *returnString = @"\r\n";
	NSString *colonString = @":";
	NSString *type, *value;
	
	NSMutableDictionary *dict = [NSMutableDictionary dictionary];
	while ([mailScanner isAtEnd] == NO)
	{
		if ([mailScanner scanUpToString:colonString intoString:&type] &&
			[mailScanner scanString:colonString intoString:NULL] &&
			[mailScanner scanUpToString:returnString intoString:&value] &&
			[mailScanner scanString:returnString intoString:NULL])
		{
			[dict setObject:value forKey:type];
		}
		else
		{
			break;
		}
	}
	
	[headerDictionary release];
	headerDictionary = [[NSDictionary alloc] initWithDictionary:dict];
	
	[body release];
	body = [[NSString alloc] initWithData:bodyData encoding:NSISO2022JPStringEncoding];
	
	return YES;
}


@end


@implementation NSString (FJDNMIMEEncode)

- (NSString *)stringByDecodingRFC2047MessageHeader
{
	NSScanner *scanner = [NSScanner scannerWithString:self];
	[scanner setCaseSensitive:NO];
	
	NSMutableString *result = [NSMutableString string];
	
	NSString *tempString;
	BOOL isScannerInEncodedString = NO;
	
	while ([scanner isAtEnd] == NO)
	{
		if (isScannerInEncodedString)
		{
			if ([scanner scanUpToString:@"?=" intoString:&tempString] && 
				[scanner scanString:@"?=" intoString:NULL])
			{
				NSArray *tempArray = [tempString componentsSeparatedByString:@"?"];
				if ([tempArray count] != 3)
				{
					// NSLog(@"String does not agree with RFC2047 format.");
					return nil;
				}
				NSStringEncoding encoding = (NSStringEncoding)CFStringConvertEncodingToNSStringEncoding(CFStringConvertIANACharSetNameToEncoding((CFStringRef)[tempArray objectAtIndex:0]));
				if (encoding == 0)
				{
					// NSLog(@"Following string encoding is not supported: %@", [tempArray objectAtIndex:0]);
					return nil;
				}
				NSString *decodedString;
				if ([[tempArray objectAtIndex:1] caseInsensitiveCompare:@"B"] == NSOrderedSame)
				{
					NSData *data = [NSData dataWithBase64EncodedString:[tempArray objectAtIndex:2]];
					decodedString = [[[NSString alloc] initWithData:data
														   encoding:encoding] autorelease];
				}
				else
				{
					// NSLog(@"Q encoding is not supported");
					return nil;
				}
				[result appendString:decodedString];
				isScannerInEncodedString = NO;
			}
			else
			{
				// NSLog(@"ERROR! Not closed by ?=.");
				return nil;
			}
		}
		else
		{
			if ([scanner scanUpToString:@"=?" intoString:&tempString])
			{
				[result appendString:tempString];
			}
			if ([scanner scanString:@"=?" intoString:NULL])
			{
				isScannerInEncodedString = YES;
			}
		}
	}
	return [NSString stringWithString:result];
}

@end

